/*
 * Copyright (C) 2014-2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "fake_json.h"
#include "mock_ual.h"
#include "test_data.h"

#include <QCoreApplication>
#include <QDebug>
#include <QDir>
#include <QTimer>

#include <cstdlib>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include <click/interface.h>
#include <click/departments-db.h>

using namespace click;
using namespace ::testing;

namespace
{

// TODO: Get rid of file-based testing and instead make unity::util::IniParser mockable
// Maintaining this list here will become tedious over time.
static const std::vector<click::Application> non_desktop_applications =
{
    {"com.ubuntu.stock-ticker-mobile", "Stock Ticker", 0.0,
        "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.stock-ticker-mobile/icons/stock_icon_48.png", "application:///com.ubuntu.stock-ticker-mobile_stock-ticker-mobile_0.3.7.66.desktop", "An awesome Stock Ticker application with all the features you could imagine", "", ""},
    {"", "Weather", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.weather/./weather64.png", "application:///com.ubuntu.weather_weather_1.0.168.desktop", "", "", ""},
    {"com.ubuntu.developer.webapps.webapp-twitter", "Twitter", 0.0,
        "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.developer.webapps.webapp-twitter/./twitter.png", "application:///com.ubuntu.developer.webapps.webapp-twitter_webapp-twitter_1.0.5.desktop", "", "", ""},
    {"com.ubuntu.music", "Music", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.music/images/music.png", "application:///com.ubuntu.music_music_1.1.329.desktop", "Ubuntu Touch Music Player", "", ""},
    {"com.ubuntu.clock", "Clock", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.clock/./clock64.png", "application:///com.ubuntu.clock_clock_1.0.300.desktop", "", "", ""},
    {"com.ubuntu.dropping-letters", "Dropping Letters", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.dropping-letters/dropping-letters.png", "application:///com.ubuntu.dropping-letters_dropping-letters_0.1.2.2.43.desktop", "", "", ""},
    {"com.ubuntu.developer.webapps.webapp-gmail", "Gmail", 0.0,
        "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.developer.webapps.webapp-gmail/./gmail.png", "application:///com.ubuntu.developer.webapps.webapp-gmail_webapp-gmail_1.0.8.desktop", "", "", ""},
    {"com.ubuntu.terminal", "Terminal", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.terminal/./terminal64.png", "application:///com.ubuntu.terminal_terminal_0.5.29.desktop", "", "", ""},
    {"com.ubuntu.calendar", "Calendar", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.calendar/./calendar64.png", "application:///com.ubuntu.calendar_calendar_0.4.182.desktop", "", "", ""},
    {"com.ubuntu.notes", "Notes", 0.0, "image://theme/notepad", "application:///com.ubuntu.notes_notes_1.4.242.desktop", "", "", ""},
    {"com.ubuntu.developer.webapps.webapp-amazon", "Amazon", 0.0,
        "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.developer.webapps.webapp-amazon/./amazon.png", "application:///com.ubuntu.developer.webapps.webapp-amazon_webapp-amazon_1.0.6.desktop", "", "", ""},
    {"com.ubuntu.shorts", "Shorts", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.shorts/./rssreader64.png", "application:///com.ubuntu.shorts_shorts_0.2.162.desktop", "", "", ""},
    {"com.ubuntu.filemanager", "File Manager", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.filemanager/./filemanager64.png", "application:///com.ubuntu.filemanager_filemanager_0.1.1.97.desktop", "", "", ""},
    {"com.ubuntu.calculator", "Calculator", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.calculator/./calculator64.png", "application:///com.ubuntu.calculator_calculator_0.1.3.206.desktop", "", "", ""},
    {"com.ubuntu.sudoku", "Sudoku", 0.0, "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.sudoku/SudokuGameIcon.png", "application:///com.ubuntu.sudoku_sudoku_1.0.142.desktop", "Sudoku Game for Ubuntu Touch", "", ""},
    {"com.ubuntu.developer.webapps.webapp-ebay", "eBay", 0.0,
        "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.developer.webapps.webapp-ebay/./ebay.png", "application:///com.ubuntu.developer.webapps.webapp-ebay_webapp-ebay_1.0.8.desktop", "", "", ""},
    {"com.ubuntu.developer.webapps.webapp-facebook", "Facebook", 0.0,
        "/usr/share/click/preinstalled/.click/users/@all/com.ubuntu.developer.webapps.webapp-facebook/./facebook.png", "application:///com.ubuntu.developer.webapps.webapp-facebook_webapp-facebook_1.0.5.desktop", "", "", ""},
    {"", "Messaging", 0.0, "image://theme/messages-app", "application:///messaging-app.desktop", "Messaging application", "/usr/share/messaging-app/assets/messaging-app-screenshot.png", ""},
    {"", "Contacts", 0.0, "image://theme/contacts-app", "application:///address-book-app.desktop", "", "", ""}
};

static click::Application desktop_application
{
    "",
    "Sample Desktop-only non-click app",
    0.0,
    "image://theme/sample-desktop-app",
    "application:///non-click-app-without-exception.desktop",
    "multiline description goes here",
    "",
    ""
};

}

namespace
{
const std::string emptyQuery{};

class ClickInterfaceTest : public ::testing::Test {
public:
    MOCK_METHOD2(manifest_callback, void(Manifest, InterfaceError));
    MOCK_METHOD2(manifests_callback, void(ManifestList, InterfaceError));
    MOCK_METHOD2(installed_callback, void(PackageSet, InterfaceError));

    std::vector<std::string> ignoredApps;
};

}

class FakeClickInterface : public click::Interface {
public:
    FakeClickInterface() {}

    MOCK_CONST_METHOD1(get_manifest_json, std::string(const std::string&));
    MOCK_CONST_METHOD0(installed_apps, std::list<std::shared_ptr<ual::Application>>());
};

TEST(ClickInterface, testIsNonClickAppFalse)
{
    EXPECT_FALSE(Interface::is_non_click_app("unknown-app.desktop"));
}

TEST(ClickInterface, testIsNonClickAppNoRegression)
{
    // Loop through and check that all filenames are non-click filenames
    // If this ever breaks, something is very very wrong.
    for (const auto& element : nonClickDesktopFiles())
    {
        EXPECT_TRUE(Interface::is_non_click_app(element));
    }
}

TEST(ClickInterface, testFindAppsInDirEmpty)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{}));
    auto results = iface.search("xyzzygy");

    EXPECT_TRUE(results.empty());
}

TEST_F(ClickInterfaceTest, testFindAppsInDirIgnoredApps)
{
    FakeClickInterface iface;
    std::list<std::shared_ptr<ual::Application>> applist{
        MockUALApplication::create(ual::AppID::parse("com.ubuntu.calculator_calculator_0.1"),
                                   std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Calculator", "The calculator", "/opt/click.ubuntu.com/foo/bar/calculator.png")}),
        MockUALApplication::create(ual::AppID::find("webbrowser-app"),
                                   std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Browser", "The browser", "webbrowser-app")}),
        MockUALApplication::create(ual::AppID::find("messaging-app"),
                                   std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Messaging", "Your messages", "messaging-app")}),
    };

    ignoredApps.push_back("messaging-app.desktop");
    ignoredApps.push_back("com.ubuntu.calculator");

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(applist));
    auto results = iface.search("", ignoredApps);
    ASSERT_EQ(1u, results.size());
}

TEST_F(ClickInterfaceTest, testFindAppsOneException)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::parse("foo_bar_1")),
                    MockUALApplication::create(ual::AppID::parse("com.ubuntu.clock_clock_0.1"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Clock", "The clock", "/opt/click.ubuntu.com/foo/bar/clock.png")}),
                        }));
    auto results = iface.search("");
    ASSERT_EQ(1u, results.size());
}

TEST_F(ClickInterfaceTest, testFindClockUsesShortAppid)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::parse("com.ubuntu.clock_clock_0.1"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Clock", "The clock", "/opt/click.ubuntu.com/foo/bar/clock.png")}),
                        }));
    auto results = iface.search("Clock");
    ASSERT_EQ(1u, results.size());
    EXPECT_EQ("appid://com.ubuntu.clock/clock/current-user-version", results.begin()->url);
}

TEST_F(ClickInterfaceTest, testFindLegacyAppUsesDeskopId)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::find("messaging-app"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Messaging", "Your messages", "messaging-app")})
                        }));
    auto results = iface.search("Messaging");
    ASSERT_EQ(1u, results.size());
    EXPECT_EQ("application:///messaging-app.desktop", results.begin()->url);
}

TEST_F(ClickInterfaceTest, testFindLegacyAppSkipped)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::find("skipped-app"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Skipped", "An app that's skipped", "skipped-app")})
                        }));
    auto results = iface.search("");
    ASSERT_EQ(0u, results.size());
}

TEST_F(ClickInterfaceTest, testFindLegacyAppNotSkipped)
{
    FakeClickInterface iface;

    std::shared_ptr<MockUALApplication::MockInfo> mockinfo{new MockUALApplication::MockInfo("Validity", "Valid app", "valid-app")};
    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::find("valid-app"),
                                               mockinfo)
                        }));
    EXPECT_CALL(*mockinfo, supportsUbuntuLifecycle()).Times(1).
        WillOnce(Return(ual::Application::Info::UbuntuLifecycle::from_raw(true)));

    auto results = iface.search("Valid");
    ASSERT_EQ(1u, results.size());
    EXPECT_EQ("application:///valid-app.desktop", results.begin()->url);
}

// test that application with a default department id key in the desktop
// file is returned when department matches
TEST_F(ClickInterfaceTest, testFindAppsWithAppWithDefaultDepartmentId)
{
    FakeClickInterface iface;

    auto mockapp = MockUALApplication::create(ual::AppID::find("address-book-app"),
                                              std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Contacts", "Address book", "address-book-app")});

    auto dept = ual::Application::Info::DefaultDepartment::from_raw("accessories");
    DefaultValue<const ual::Application::Info::DefaultDepartment&>::Set(dept);

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{mockapp}));

    auto depts_db = std::make_shared<click::DepartmentsDb>(":memory:");
    auto results = iface.search("", ignoredApps, "accessories", depts_db);

    ASSERT_EQ(1u, results.size());
    EXPECT_EQ("Contacts", results.begin()->title);
}

TEST_F(ClickInterfaceTest, testFindAppsWithAppWithDefaultDepartmentIdOverriden)
{
    FakeClickInterface iface;

    auto mockapp = MockUALApplication::create(ual::AppID::find("address-book-app"),
                                              std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Contacts", "Address book", "address-book-app")});

    auto dept = ual::Application::Info::DefaultDepartment::from_raw("accessories");
    DefaultValue<const ual::Application::Info::DefaultDepartment&>::Set(dept);

    EXPECT_CALL(iface, installed_apps()).
        WillRepeatedly(Return(std::list<std::shared_ptr<ual::Application>>{mockapp}));

    auto depts_db = std::make_shared<click::DepartmentsDb>(":memory:");

    depts_db->store_department_name("utilities", "", "Utilities");
    depts_db->store_department_name("accessories", "", "Accessories");
    depts_db->store_department_mapping("utilities", "");
    depts_db->store_department_mapping("accessories", "");

    auto results = iface.search("", ignoredApps, "utilies", depts_db);
    ASSERT_EQ(0u, results.size());

    // address book applicaton moved to utilities
    depts_db->store_package_mapping("address-book-app.desktop", "utilities");
    results = iface.search("", ignoredApps, "utilities", depts_db);

    ASSERT_EQ(1u, results.size());
    EXPECT_EQ("Contacts", results.begin()->title);
}

TEST(ClickInterface, testFindAppsInDirSorted)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::parse("com.ubuntu.clock_clock_0.1"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Clock", "The clock", "/opt/click.ubuntu.com/foo/bar/clock.png")}),
                    MockUALApplication::create(ual::AppID::find("address-book-app"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Contacts", "Address book", "address-book-app")}),
                     MockUALApplication::create(ual::AppID::parse("com.ubuntu.stock-ticker-mobile_stockticker_0.1"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Stock Ticker", "A stock ticker.", "/opt/click.ubuntu.com/foo/bar/stock_icon_48.png")}),
                       }));

    auto results = iface.search("ock");

    const std::vector<click::Application> expected_results = {
        {"com.ubuntu.clock", "Clock", 0.0,
         "/opt/click.ubuntu.com/foo/bar/clock.png",
         "appid://com.ubuntu.clock/clock/current-user-version",
         "The clock", "", ""},
        {"com.ubuntu.stock-ticker-mobile", "Stock Ticker", 0.0,
         "/opt/click.ubuntu.com/foo/bar/stock_icon_48.png",
         "appid://com.ubuntu.stock-ticker-mobile/stockticker/current-user-version",
         "A stock ticker.", "", ""},
    };
    EXPECT_EQ(expected_results, results);
}

TEST(ClickInterface, testSortApps)
{
    std::vector<click::Application> apps = {
        {"", "Sudoku", 0.0, "", "", "", "", ""},
        {"", "eBay", 0.0, "", "", "", "", ""},
        {"", "Facebook", 0.0, "", "", "", "", ""},
        {"", "Messaging", 0.0, "", "", "", "", ""},
        {"", "Contacts", 0.0, "", "", "", "", ""},
    };

    std::vector<click::Application> expected = {
        {"", "Contacts", 0.0, "", "", "", "", ""},
        {"", "eBay", 0.0, "", "", "", "", ""},
        {"", "Facebook", 0.0, "", "", "", "", ""},
        {"", "Messaging", 0.0, "", "", "", "", ""},
        {"", "Sudoku", 0.0, "", "", "", "", ""},
    };

    ASSERT_EQ(setenv(Configuration::LANGUAGE_ENVVAR, "en_US.UTF-8", 1), 0);
    EXPECT_EQ(expected, click::Interface::sort_apps(apps));
    ASSERT_EQ(unsetenv(Configuration::LANGUAGE_ENVVAR), 0);
}

TEST(ClickInterface, testSortAppsWithDuplicates)
{
    std::vector<click::Application> apps = {
        {"com.sudoku.sudoku", "Sudoku", 0.0, "", "", "", "", ""},
        {"com.canonical.sudoku", "Sudoku", 0.0, "", "", "", "", ""},
    };

    std::vector<click::Application> expected = {
        {"com.canonical.sudoku", "Sudoku", 0.0, "", "", "", "", ""},
        {"com.sudoku.sudoku", "Sudoku", 0.0, "", "", "", "", ""},
    };

    ASSERT_EQ(setenv(Configuration::LANGUAGE_ENVVAR, "en_US.UTF-8", 1), 0);
    EXPECT_EQ(expected, click::Interface::sort_apps(apps));
    ASSERT_EQ(unsetenv(Configuration::LANGUAGE_ENVVAR), 0);
}

TEST(ClickInterface, testSortAppsWithAccents)
{
    std::vector<click::Application> apps = {
        {"", "Robots", 0.0, "", "", "", "", ""},
        {"", "Æon", 0.0, "", "", "", "", ""},
        {"", "Contacts", 0.0, "", "", "", "", ""},
        {"", "Über", 0.0, "", "", "", "", ""},
    };

    std::vector<click::Application> expected = {
        {"", "Æon", 0.0, "", "", "", "", ""},
        {"", "Contacts", 0.0, "", "", "", "", ""},
        {"", "Robots", 0.0, "", "", "", "", ""},
        {"", "Über", 0.0, "", "", "", "", ""},
    };

    ASSERT_EQ(setenv(Configuration::LANGUAGE_ENVVAR, "en_US.UTF-8", 1), 0);
    EXPECT_EQ(expected, click::Interface::sort_apps(apps));
    ASSERT_EQ(unsetenv(Configuration::LANGUAGE_ENVVAR), 0);
}

TEST(ClickInterface, testSortAppsMixedCharsets)
{
    std::vector<click::Application> apps = {
        {"", "Robots", 0.0, "", "", "", "", ""},
        {"", "汉字", 0.0, "", "", "", "", ""},
        {"", "漢字", 0.0, "", "", "", "", ""},
        {"", "Über", 0.0, "", "", "", "", ""},
    };

    std::vector<click::Application> expected = {
        {"", "汉字", 0.0, "", "", "", "", ""},
        {"", "漢字", 0.0, "", "", "", "", ""},
        {"", "Robots", 0.0, "", "", "", "", ""},
        {"", "Über", 0.0, "", "", "", "", ""},
    };

    ASSERT_EQ(setenv(Configuration::LANGUAGE_ENVVAR, "zh_CN.UTF-8", 1), 0);
    EXPECT_EQ(expected, click::Interface::sort_apps(apps));
    ASSERT_EQ(unsetenv(Configuration::LANGUAGE_ENVVAR), 0);
}

TEST_F(ClickInterfaceTest, testFindAppByKeyword)
{
    FakeClickInterface iface;

    auto mockapp = MockUALApplication::create(ual::AppID::parse("com.ubuntu.camera_camera_5"),
                                              std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Cámara", "La cámara", "/opt/click.ubuntu.com/foo/bar/camera.png")});

    std::vector<std::string> keys{"foo", "bar", "rss", "baz"};
    auto mockkw = ual::Application::Info::Keywords::from_raw(keys);
    DefaultValue<const ual::Application::Info::Keywords&>::Set(mockkw);

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{mockapp}));

    auto results = iface.search("rss");

    EXPECT_EQ(1u, results.size());
}

TEST_F(ClickInterfaceTest, testFindAppByKeywordCaseInsensitive)
{
    FakeClickInterface iface;

    auto mockapp = MockUALApplication::create(ual::AppID::parse("com.ubuntu.camera_camera_5"),
                                              std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Cámara", "La cámara", "/opt/click.ubuntu.com/foo/bar/camera.png")});

    std::vector<std::string> keys{"foo", "bar", "rss", "baz"};
    auto mockkw = ual::Application::Info::Keywords::from_raw(keys);
    DefaultValue<const ual::Application::Info::Keywords&>::Set(mockkw);

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{mockapp}));

    auto results = iface.search("RsS");

    EXPECT_EQ(1u, results.size());
}

TEST_F(ClickInterfaceTest, testFindAppAccented)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::parse("com.ubuntu.camera_camera_5"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Cámara", "La cámara", "/opt/click.ubuntu.com/foo/bar/camera.png")}),
                        }));

    auto results = iface.search("Cámara");

    EXPECT_EQ(1u, results.size());
}

TEST_F(ClickInterfaceTest, testFindAppAccented2)
{
    FakeClickInterface iface;

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::parse("com.ubuntu.camera_camera_5"),
                                               std::shared_ptr<MockUALApplication::MockInfo>{new MockUALApplication::MockInfo("Cámara", "La cámara", "/opt/click.ubuntu.com/foo/bar/camera.png")}),
                        }));

    auto results = iface.search("Camara");

    EXPECT_EQ(1u, results.size());
}


TEST(ClickInterface, testIsIconIdentifier)
{
    EXPECT_TRUE(Interface::is_icon_identifier("contacts-app"));
    EXPECT_FALSE(Interface::is_icon_identifier(
                    "/usr/share/unity8/graphics/applicationIcons/contacts-app@18.png"));
}

TEST(ClickInterface, testAddThemeScheme)
{
    EXPECT_EQ("image://theme/contacts-app", Interface::add_theme_scheme("contacts-app"));
    EXPECT_EQ("/usr/share/unity8/graphics/applicationIcons/contacts-app@18.png",
              Interface::add_theme_scheme("/usr/share/unity8/graphics/applicationIcons/contacts-app@18.png"));
}

TEST(ClickInterface, testManifestFromJsonOneApp)
{
    Manifest m = manifest_from_json(FAKE_JSON_MANIFEST_ONE_APP);
    ASSERT_EQ(m.first_app_name, "fake-app");
    ASSERT_TRUE(m.has_any_apps());
    ASSERT_FALSE(m.has_any_scopes());
}

TEST(ClickInterface, testManifestFromJsonOneScope)
{
    Manifest m = manifest_from_json(FAKE_JSON_MANIFEST_ONE_SCOPE);
    ASSERT_EQ(m.first_scope_id, "com.example.fake-scope_fake-scope-hook");
    ASSERT_FALSE(m.has_any_apps());
    ASSERT_TRUE(m.has_any_scopes());
}

TEST(ClickInterface, testManifestFromJsonOneAppOneScope)
{
    Manifest m = manifest_from_json(FAKE_JSON_MANIFEST_ONE_APP_ONE_SCOPE);
    ASSERT_EQ(m.first_app_name, "fake-app");
    ASSERT_EQ(m.first_scope_id, "com.example.fake-1app-1scope_fake-scope-hook");
    ASSERT_TRUE(m.has_any_apps());
    ASSERT_TRUE(m.has_any_scopes());
}

TEST(ClickInterface, testManifestFromJsonTwoAppsTwoScopes)
{
    Manifest m = manifest_from_json(FAKE_JSON_MANIFEST_TWO_APPS_TWO_SCOPES);
    ASSERT_EQ(m.first_app_name, "fake-app1");
    ASSERT_EQ(m.first_scope_id, "com.example.fake-2apps-2scopes_fake-scope-hook1");
    ASSERT_TRUE(m.has_any_apps());
    ASSERT_TRUE(m.has_any_scopes());
}

TEST_F(ClickInterfaceTest, testGetManifestForAppParseError)
{
    FakeClickInterface iface;
    EXPECT_CALL(iface, get_manifest_json(_)).
        Times(1).
        WillOnce(Return("INVALID JSON"));
    EXPECT_CALL(*this, manifest_callback(_, InterfaceError::ParseError));
    iface.get_manifest_for_app(FAKE_PACKAGENAME, [this](Manifest manifest,
                                                        InterfaceError error){
                                   manifest_callback(manifest, error);
                               });
}

TEST_F(ClickInterfaceTest, testGetManifestForAppIsRemovable)
{
    FakeClickInterface iface;
    EXPECT_CALL(iface, get_manifest_json(_)).
        Times(1).
        WillOnce(Return(FAKE_JSON_MANIFEST_REMOVABLE));
    iface.get_manifest_for_app(FAKE_PACKAGENAME, [](Manifest manifest,
                                                    InterfaceError error){
                                   ASSERT_TRUE(error == InterfaceError::NoError);
                                   ASSERT_TRUE(manifest.removable);
                               });
}

TEST_F(ClickInterfaceTest, testGetManifestForAppIsNotRemovable)
{
    FakeClickInterface iface;
    EXPECT_CALL(iface, get_manifest_json(_)).
        Times(1).
        WillOnce(Return(FAKE_JSON_MANIFEST_NONREMOVABLE));
    iface.get_manifest_for_app(FAKE_PACKAGENAME, [](Manifest manifest,
                                                    InterfaceError error){
                                   ASSERT_TRUE(error == InterfaceError::NoError);
                                   ASSERT_FALSE(manifest.removable);
                               });
}

TEST_F(ClickInterfaceTest, testGetInstalledPackagesParsed)
{
    FakeClickInterface iface;
    PackageSet expected{{"foo.bar", "0.1"}, {"baz.foo", "0.2"}};

    EXPECT_CALL(iface, installed_apps()).Times(1).
        WillOnce(Return(std::list<std::shared_ptr<ual::Application>>{
                    MockUALApplication::create(ual::AppID::parse("foo.bar_foo_0.1")),
                    MockUALApplication::create(ual::AppID::parse("baz.foo_foo_0.2"))
                 }));
    iface.get_installed_packages([expected](PackageSet package_names, InterfaceError error){
            ASSERT_EQ(error, InterfaceError::NoError);
            ASSERT_EQ(package_names, expected);
    });
}

