/**
 * VampirTrace
 * http://www.tu-dresden.de/zih/vampirtrace
 *
 * Copyright (c) 2005-2013, ZIH, TU Dresden, Federal Republic of Germany
 *
 * Copyright (c) 1998-2005, Forschungszentrum Juelich, Juelich Supercomputing
 *                          Centre, Federal Republic of Germany
 *
 * See the file COPYING in the package base directory for details
 **/

package widgets;
import help.Config;
import help.MyListRenderer;
import help.MyTableRenderer;

import javax.swing.BoxLayout;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ToolTipManager;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.util.ArrayList;


/*****
 * This class is a component that consists of a table and a listbox to 
 * to manage papi counter
 */
public class CounterWidget extends javax.swing.JPanel{

	private static final long serialVersionUID = 1L;
	
	private JList dest = null;
	private JTable src = null;
	private ArrayList<String> defaultvals = new ArrayList<String>();
	private JCheckBox check = null;
	private JLabel error = null;
	private DefaultListModel destModel = null;
	private DefaultTableModel srcModel_all = null;
	private DefaultTableModel srcModel = null;
	private JButton button = null;
	private JButton button1 = null;
	private TableRowSorter<TableModel> sorter = null;
	private boolean noPapiEventChooser = true;
	private MyTableRenderer cellRenderer = new MyTableRenderer();
	
	/****
	 * Instantiate widget class
	 */
	public CounterWidget()
	{
		initWidget();
	}
	
	/****
	 * initialize all Components of this Widget
	 */
	public void initWidget()
	{
		error = new JLabel("");
		destModel = new DefaultListModel();
		srcModel_all = new DefaultTableModel();
		srcModel = new DefaultTableModel();
		button = new JButton(" -> ");
		button1 = new JButton(" <- ");
		sorter = null;
		noPapiEventChooser = true;
		
		
		src = new JTable(srcModel){
			/**
			 * serial VersionId
			 */
			private static final long serialVersionUID = -6322589002946229578L;

			public boolean isCellEditable(int rowIndex, int vColIndex) { return false; } 
		};
		
		src.setDefaultRenderer(Object.class, cellRenderer);
		dest = new JList(destModel);
		src.getTableHeader().setReorderingAllowed(false);
		
		
		dest.setCellRenderer(new MyListRenderer());
		
		
		ToolTipManager.sharedInstance().unregisterComponent(src);
		ToolTipManager.sharedInstance().unregisterComponent(src.getTableHeader());


		
		srcModel.addColumn("Name:");
		srcModel.addColumn("Available:");
		srcModel.addColumn("Derived:");
		srcModel.addColumn("Description:");
		
		resizeColumns();
		
		srcModel_all.addColumn("Name:");
		srcModel_all.addColumn("Available:");
		srcModel_all.addColumn("Derived:");
		srcModel_all.addColumn("Description:");
		
		
		dest.setLayoutOrientation(JList.VERTICAL);
		
		sorter = new TableRowSorter<TableModel>(src.getModel());
		src.setRowSorter(sorter);

		
		src.setCellSelectionEnabled(false);
		src.setRowSelectionAllowed(true);
		 
		   
		JScrollPane listScroller = new JScrollPane(src);
		listScroller.setPreferredSize(new Dimension(190, 250));
		listScroller.setMinimumSize(new Dimension(190, 250));
		listScroller.setVerticalScrollBarPolicy( JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		listScroller.setHorizontalScrollBarPolicy( JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		JScrollPane listScroller2 = new JScrollPane(dest);
		listScroller2.setPreferredSize(new Dimension(220, 250));
		
		
		check = new JCheckBox("Hide unavailable counters");
		check.setSelected(true);
		
		check.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				if(check.isSelected())
					src.setModel(srcModel);
				else
					src.setModel(srcModel_all);
				
				 sorter.setModel(src.getModel());
				
				resizeColumns();
			}
		});
		
		src.addMouseListener(new MouseListener(){

			public void mouseClicked(MouseEvent arg0) {
				if(arg0.getClickCount() >= 2)
					addOrRemoveCounter("add");				
			}

			public void mouseEntered(MouseEvent arg0) {
				;
			}


			public void mouseExited(MouseEvent arg0) {
				;
			}


			public void mousePressed(MouseEvent arg0) {
				;
			}


			public void mouseReleased(MouseEvent arg0) {
				;
			}
			
		});
		dest.addMouseListener(new MouseListener(){


			public void mouseClicked(MouseEvent arg0) {
				if(arg0.getClickCount() >= 2)
					addOrRemoveCounter("remove");
			}


			public void mouseEntered(MouseEvent arg0) {
				;
			}


			public void mouseExited(MouseEvent arg0) {
			;
			}


			public void mousePressed(MouseEvent arg0) {
			;
			}
			public void mouseReleased(MouseEvent arg0) {
			;	
			}
			
		});
		src.addKeyListener(new KeyListener(){
			public void keyPressed(KeyEvent arg0) {

				if(arg0.getKeyCode() == 10)
					addOrRemoveCounter("add");

			}

			public void keyReleased(KeyEvent arg0) {
				
			}


			public void keyTyped(KeyEvent arg0) {

			}
			
		});
		dest.addKeyListener(new KeyListener(){
			public void keyPressed(KeyEvent arg0) {

				if(arg0.getKeyCode() == 10 || arg0.getKeyCode() == 127)
					addOrRemoveCounter("remove");
			}

			public void keyReleased(KeyEvent arg0) {
				
			}


			public void keyTyped(KeyEvent arg0) {

			}

		});


		GridBagConstraints c = new GridBagConstraints();
		this.setLayout(new GridBagLayout());
		
		JPanel buttonPane = new JPanel();
		buttonPane.setLayout(new BoxLayout(buttonPane,BoxLayout.PAGE_AXIS));
		
		buttonPane.add(button);
		buttonPane.add(button1);
		
		c.fill = GridBagConstraints.HORIZONTAL;
		c.gridx = 0;
		c.gridy = 0;
		c.gridwidth = 2;
		c.insets = new Insets(0,5,0,5);
		c.anchor = GridBagConstraints.NORTH;
		c.weightx = 1;
		this.add(listScroller,c);
		c.weightx = 0;
		c.gridwidth = 1;
		c.gridx = 2;
		c.anchor = GridBagConstraints.CENTER;
		this.add(buttonPane,c);
		c.anchor = GridBagConstraints.NORTH;
		c.gridx = 3;
		this.add(listScroller2,c);
		c.gridx = 0;
		c.gridy = 1;
		c.gridwidth = 1;
		c.anchor = GridBagConstraints.LAST_LINE_START;
		this.add(check,c);
		c.gridx = 1;
		c.gridy = 1;
		c.gridwidth=3;
		c.anchor = GridBagConstraints.LAST_LINE_END;
		this.add(error,c);
		
		getCounters();
		checkPapiEventChooser();
		getValue();
		button.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				addOrRemoveCounter("add");
			}
		});
		
		button1.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				addOrRemoveCounter("remove");
			}
		});
		
		
		src.getSelectionModel().addListSelectionListener(new ListSelectionListener(){
			public void valueChanged(ListSelectionEvent e){
				dest.clearSelection();
			}
		});

		
		dest.addListSelectionListener(new ListSelectionListener(){
			public void valueChanged(ListSelectionEvent e) {
				src.clearSelection();
			}
		});
	
	
		MouseAdapter listener = new ReorderListener(dest);
	    
		dest.addMouseListener(listener);
	    dest.addMouseMotionListener(listener);

	    
	}

	/****
	 * move counter to listbox or remove them from listbox
	 */
	public void addOrRemoveCounter(String type)
	{
		if(type.endsWith("add"))
		{
			int[] val = this.src.getSelectedRows();
			
			for(int i = 0; i < val.length; i++ )
			{
				if(!this.destModel.contains(this.src.getValueAt(val[i], 0)))
				{
					if(src.getValueAt(val[i], 1).toString().toLowerCase().trim().compareTo("no")==0)
						JOptionPane.showMessageDialog(null, "Counter \""+src.getValueAt(val[i], 0)+"\" not available!", "Warning", JOptionPane.WARNING_MESSAGE);
					else
						this.destModel.addElement(this.src.getValueAt(val[i], 0));
				}
			}
			this.src.clearSelection();
		}
		
		if(type.equals("remove"))
		{
			int[] valdest = this.dest.getSelectedIndices();
			for(int i = valdest.length-1; i >= 0; i-- )
			{
				this.destModel.remove(valdest[i]);
			}
			this.dest.clearSelection();
		}
		
		if(!noPapiEventChooser)
			checkCounter(0);
	}
	
	/****
	 * check if the choosen papi counter are valid with the help of papi_event_chooser
	 * if counter are not valid they are painted red.
	 * @param offset is used by this recursive method. Start this method with an offset of 0;
	 * @return returns true;
	 */
	private boolean checkCounter(int offset)
	{
		if(destModel.getSize() == 0 || noPapiEventChooser)
			return true;
		
		try{
			Process p = null;
			String call = "papi_event_chooser ";
			
			
			if(destModel.get(0).toString().trim().startsWith("PAPI"))
				call += "PRESET ";
			else
				call += "NATIVE ";
			
			
			int colBorder = 0;
			boolean error = false;
			for(int i = 0; i<destModel.getSize()-offset;i++)
			{
				call += destModel.get(i).toString()+" ";
				colBorder = i;
			}

			p = Runtime.getRuntime().exec(call);

			 BufferedReader br = new BufferedReader(new InputStreamReader(p.getErrorStream()));
			 if(br.readLine() != null)
				 error = true;
			 br.close();
			 p.destroy();
			 if(error)
			 {
				 checkCounter(offset+1);
			 }
			 else
			 {
				 ((MyListRenderer)dest.getCellRenderer()).repaint();
				 ((MyListRenderer)dest.getCellRenderer()).setColorBorder(colBorder);
				 
				 if(offset != 0)
					 this.error.setText("Only the counters highlighted in green can be measured at the same time!");
				 else
					 this.error.setText("");
			 }
			 
		}catch(Exception e)
		{
			Config.errorHandler(e);
		}
		return true;
	}
	
	/****
	 * check if the papi_event_chooser is installed. and sets noPapiEventChooser to true or false
	 */
	private void checkPapiEventChooser()
	{
		try{
			String call = "papi_event_chooser PRESET ";
			String getCounter = "";
			for(int i=0;srcModel.getRowCount() > i && (getCounter.equals("") || getCounter == null);i++)
			{
				if(srcModel.getValueAt(i, 1).toString().toLowerCase().trim().equals("yes"))
				{
					getCounter = srcModel.getValueAt(i, 0).toString();
				}
			}
				
			call += getCounter;
			Process p = Runtime.getRuntime().exec(call);
			BufferedReader br = new BufferedReader(new InputStreamReader(p.getErrorStream()));
			String line;
			line = br.readLine();
			
			if(line == null)
			{

				noPapiEventChooser = false;
				((MyListRenderer)dest.getCellRenderer()).setColorIt(true);
				
			}else{
				noPapiEventChooser = true;
				((MyListRenderer)dest.getCellRenderer()).setColorIt(false);
				
			}
			br.close();
			p.destroy();
		}catch(Exception e)
		{
			System.out.println("no papi_event_chooser detected. Can't validate papi counters.");
			noPapiEventChooser = true;
			//	Config.errorHandler(e);
		}
	}
	
	/****
	 * Loads papi counter with papi_avail and put them into the table.
	 */
	private void getCounters()
	{
		try{
			   Process p = Runtime.getRuntime().exec("papi_avail");  

			   if(p.getErrorStream().available() == 0)
			   {
				   BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()));   
				   String line;
				   String[] help = new String[4];
				   while ((line = br.readLine()) != null) {  
				     
				     while(line.contains("  ")){
				    	 line = line.replaceAll("  ", " ");
				     }
					 help = line.split(" ",5); //tab
					 String[] rowData = new String[4];
					   if(help[0].startsWith("PAPI_"))
					   {
						   rowData[0] = help[0];
						   rowData[1] = help[2];
						   rowData[2] = help[3];
						   rowData[3] = help[4];
						  
						   srcModel_all.addRow(rowData);

						   if(help[2].trim().toLowerCase().compareTo("yes") == 0)
						   		srcModel.addRow(rowData);
					   }
				   }
				   checkCounter(0);
				   br.close();
			   }
		}catch(Exception ex)
		{
			System.out.println("no papi counter available");
			//Config.errorHandler(ex);
		}
	}
	
	
	/***
	 * get all Counter from the listbox 
	 * @return colon-separated list of all chosen counters
	 */
	public String getValue()
	{
		String erg = "";
		for(int i=0;i<destModel.getSize();i++)
		{
			erg += ":"+destModel.elementAt(i);
		}
		
		if(erg.length() > 0)
			erg = erg.substring(1);
		return erg;
	}
	
	
	/****
	 * set chosen counter 
	 * @param val colon-separated list of counter
	 */
	public void setValue(String val)
	{
		try{
		if(val != null && !val.trim().equals(""))
		{
			String[] values = val.split(":");
			int count = 0;
			boolean ready =false;
			for(int i=0;i<srcModel.getRowCount();i++)
			{
				for(int k=0;k<values.length;k++)
				{
					if(values[k].trim().compareTo(srcModel.getValueAt(i, 0).toString().trim()) == 0)
					{
						count++;
						destModel.addElement(values[k]);
						defaultvals.add(values[k]);
						values[k] = "";
						
						if(count == values.length)
						{
							ready = true;
						}
							break;
					}
				}
				if(ready)
				{
					break;
				}
			}
			
			if(!ready)
			{
				JOptionPane.showMessageDialog(null, "Some default counter couldn't be set.", "Warning", JOptionPane.WARNING_MESSAGE);
			}
		}
		}catch(Exception ex)
		{
			Config.errorHandler(ex);
		}
	}
	
	/***
	 * method to format the output of the counter table
	 */
	private void resizeColumns()
	{
		TableColumnModel columns = src.getColumnModel();
		columns.getColumn(1).setMaxWidth(0);
		columns.getColumn(1).setMinWidth(0);
		columns.getColumn(1).setPreferredWidth(0);
		columns.getColumn(1).setResizable(false);
		
		columns.getColumn(0).setMaxWidth(120);
		columns.getColumn(0).setMinWidth(120);
		columns.getColumn(0).setPreferredWidth(120);
		
		columns.getColumn(2).setMaxWidth(70);
		columns.getColumn(2).setMinWidth(70);
		columns.getColumn(2).setPreferredWidth(70);
		
		//columns.getColumn(2).setMaxWidth(70);
		columns.getColumn(3).setMinWidth(0);
		columns.getColumn(3).setPreferredWidth(100);
	}
	
	/****
	 * method returns all components used in this widget. 
	 * @return a list of all components used in this widget
	 */
	public ArrayList<Component> getAllComponents()
	{
		ArrayList<Component> erg = new ArrayList<Component>();
		erg.add(src);
		erg.add(check);
		erg.add(dest);
		erg.add(button);
		erg.add(error);
		return erg;
	}
	
	/***
	 * method to set this widget to default settings
	 */
	public void setDefault()
	{
		error.setText("");
		check.setSelected(true);
		src.setModel(srcModel);
		sorter.setModel(src.getModel());
		
		destModel.removeAllElements();
		for(int i=0;i<defaultvals.size();i++)
		{
			destModel.addElement(defaultvals.get(i));
		}
		
	}
	
	/***
	 * method to set widget to given settings
	 * @param val String that represent the settings
	 */
	public void load(String val)
	{
		String[] values = val.split(":");
		destModel.clear();
		for(int i=0;i<values.length;i++)
		{
			if(!values[i].trim().isEmpty())
				destModel.addElement(values[i]);
		}
		if(!noPapiEventChooser)
			checkCounter(0);
	}
	/***
	 * ReorderListener inherits from MouseAdapter and manage the drag and drop behavior of the listbox elements in the CounterWidget class
	 */
 	public class ReorderListener extends MouseAdapter {
		 
		   private JList list;
		   private int pressIndex = 0;
		   private int releaseIndex = 0;
		 
		   public ReorderListener(JList list) {
		      this.list = list;
		   }
		 
		   @Override
		   public void mousePressed(MouseEvent e) {
		      pressIndex = list.locationToIndex(e.getPoint());
		   }
		 
		   @Override
		   public void mouseReleased(MouseEvent e) {
		      releaseIndex = list.locationToIndex(e.getPoint());
		      if (releaseIndex != pressIndex && releaseIndex != -1) {
		         reorder();
		      }
		   }
		 
		   @Override
		   public void mouseDragged(MouseEvent e) {
		      mouseReleased(e);
		      pressIndex = releaseIndex;      
		   }
		 
		   private void reorder() {
		      DefaultListModel model = (DefaultListModel) list.getModel();
		      Object dragee = model.elementAt(pressIndex);
		      model.removeElementAt(pressIndex);
		      model.insertElementAt(dragee, releaseIndex);
		      if(!noPapiEventChooser)
					checkCounter(0);
		   }
		}
 	
 	public void setDisabled()
 	{
 		for(int i=0;i<this.getComponentCount();i++)
 			getComponent(i).setEnabled(false);
 		cellRenderer.setDisabled();
 		((MyListRenderer)dest.getCellRenderer()).setDisabled();
 		dest.setEnabled(false);
 		src.setEnabled(false);
 		button.setEnabled(false);
 		button1.setEnabled(false);
 	}
 	
 	
 	  

}