package Jenkins::i18n::Assertions;
use warnings;
use strict;
use Set::Tiny 0.04;
use Exporter 'import';

our @EXPORT_OK = qw(has_empty is_jelly_file has_hudson can_ignore);

=pod

=head1 NAME

Jenkins::i18n::Assertions: execute assertions on files, Properties keys and
keys values.

=head1 SYNOPSIS

    use Jenkins::i18n::Assertions qw(is_jelly_file);

    if (is_jelly_file($filename)) {
        # do something
    }

=head1 DESCRIPTION

This module provides several assertions required to process the Jenkins
internationalization related files.

=head1 FUNCTIONS

=head2 has_empty

Some Properties have keys with empty values (for any reason).

When the value of the Property key is empty, but that is expected, that's fine.
The convention in those cases is that the Property key has a C<empty> string as
part of it's name.

This function just tests that. The test is B<case insensitive>.

Expects as parameter a key Property.

Returns 1, if there is a match, 0 otherwise.

=cut

my $empty_regex = qr/empty/i;

sub has_empty {
    my $property_value = shift;
    return ( $property_value =~ $empty_regex ) ? 1 : 0;
}

=head2 is_jelly_file

Tests if a file name contains C<.jelly> as an file extension. The match B<is>
case sensitive.

Expects as parameter a file name (maybe be a full path).

Returns 1, if there is a match, 0 otherwise.

=cut

my $jelly_regex = qr/.jelly$/;

sub is_jelly_file {
    my $filename = shift;
    return ( $filename =~ $jelly_regex ) ? 1 : 0;
}

=head2 has_hudson

Tests if a Property key value has the C<Hudson> string on it. The match B<is>
case sensitive.

Expects as parameter a Property key value.

Returns 1, if there is a match, 0 otherwise.

=cut

my $hudson_regex = qr/Hudson/;

sub has_hudson {
    my $property_value = shift;
    return ( $property_value =~ $hudson_regex ) ? 1 : 0;
}

=head2 can_ignore

Some Properties keys have values that don't have to be translated, like URL's
and proper nouns. This functions tests for those known cases.

Expects as parameter a Properties key value.

Returns 1, if there is a match, 0 otherwise.

=cut

my $ignore_same = Set::Tiny->new(
    (
        'https://www.jenkins.io/redirect/log-levels',
        'Maven',
        'Jenkins',
        'JDK',
        'Javadoc',
        'https://www.jenkins.io/redirect/fingerprint',
        'https://www.jenkins.io/redirect/search-box'
    )
);

sub can_ignore {
    my $property_value = shift;
    return $ignore_same->has($property_value);
}

1;
__END__

=head1 SEE ALSO

=over

=item *

L<Set::Tiny>

=back

=head1 AUTHOR

Alceu Rodrigues de Freitas Junior, E<lt>arfreitas@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022 of Alceu Rodrigues de Freitas Junior,
E<lt>arfreitas@cpan.orgE<gt>

This file is part of Jenkins Translation Tool project.

Jenkins Translation Tool is free software: you can redistribute it and/or
modify it under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your option)
any later version.

Jenkins Translation Tool is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
Jenkins Translation Tool. If not, see (http://www.gnu.org/licenses/).

The original C<translation-tool.pl> script was licensed through the MIT
License, copyright (c) 2004-, Kohsuke Kawaguchi, Sun Microsystems, Inc., and a
number of other of contributors. Translations files generated by the Jenkins
Translation Tool CLI are distributed with the same MIT License.

=cut
